/*
* Copyright 2019-2023 Amazon.com, Inc. or its affiliates. All Rights Reserved.
*
* SPDX-License-Identifier:  GPL-2.0+
*/

#include <malloc.h>
#include <ubi_uboot.h>
#include <ring_config.h>

#define COCOA_UBI_CONFIG_VOLUME_NAME "bootconfig"

#define DEBUG
#undef DEBUG

#ifdef DEBUG
#define debugf(fmt, args...) do { printf("%s(): ", __func__); printf(fmt, ##args); } while (0)
#else
#define debugf(fmt, args...)
#endif

#define STR_HELPER(x) #x
#define STR(x) STR_HELPER(x)

#ifndef COCOA_BOOTCONFIG_SIZE
#error Need to define COCOA_BOOTCONFIG_SIZE
#endif

#ifndef COCOA_DATA_PARTITION
#error Need to specify data partition name
#endif

/* Forward declarations for functions to recreate customer partition */
int ubi_remove_vol(char *volume);
int ubi_create_vol(char *volume, int64_t size, int dynamic);

// Reset ring_config to default state
void ring_config_reset(ring_config *config)
{
	memset(config, 0, sizeof(ring_config));
}

void get_ring_config(ring_config *config)
{
	int err = 0;
	ring_config_reset(config);

	// Does ring_config ubi partition exist?
	uint8_t buffer[sizeof(ring_config)];
	if (ubi_volume_read(COCOA_UBI_CONFIG_VOLUME_NAME, (char *)buffer, sizeof(ring_config))) {
		debugf("ring config does not exist\n");

		// Recreate the volume in case it's gone missing
		err = ubi_create_vol(COCOA_UBI_CONFIG_VOLUME_NAME, COCOA_BOOTCONFIG_SIZE, 1);
		if (-ENOSPC == err) {
			debugf("Recreate %s failed, remove %s and retry\n", COCOA_UBI_CONFIG_VOLUME_NAME, STR(COCOA_DATA_PARTITION));
			ubi_remove_vol(STR(COCOA_DATA_PARTITION));
			err = ubi_create_vol(COCOA_UBI_CONFIG_VOLUME_NAME, COCOA_BOOTCONFIG_SIZE, 1);
		}
		debugf("Recreate %s %s err=%d\n", COCOA_UBI_CONFIG_VOLUME_NAME, err == 0 ? "success" : "failed", err);
		return;
	}

	// Copy header
	memcpy((uint8_t*)config, buffer, sizeof(config->header));

	// Check if header is valid
	if (config->header.magic != COCOA_RING_CONFIG_MAGIC ||
			config->header.length > sizeof(ring_config) ||
			config->header.version > COCOA_RING_CONFIG_VERSION) {
		debugf("ring config header not valid\n");
		ring_config_reset(config);
		return;
	}

	// Copy the rest less the checksum
	// We are handling the case that new data has been added to the structure,
	// so the length in the header is smaller than the structure size
	ssize_t size = config->header.length - sizeof(config->header) - sizeof(config->checksum);
	if (size <= 0) {
		debugf("Invalid ring config header length\n");
		ring_config_reset(config);
		return;
	}
	memcpy(((uint8_t*)config) + sizeof(config->header), buffer + sizeof(config->header), size);

	// Copy checksum
	memcpy(&config->checksum, buffer + config->header.length - sizeof(config->checksum), sizeof(config->checksum));

	size_t sz = config->header.length - sizeof(config->checksum);
	unsigned char* buf = (unsigned char*)config;

	// Verify checksum
	uint32_t crc = crc32_wd(0, buf, sz, CHUNKSZ_CRC32);
	if (config->checksum != crc) {
		debugf("ring config checksum failure %x vs %x\n", config->checksum, crc);
		ring_config_reset(config);
		return;
	}

	debugf("\nSuccessfully got ring config %d\n", config->boot.recoveryMode);
}

int set_ring_config(ring_config *config)
{
	int err = 0;

	config->header.magic = COCOA_RING_CONFIG_MAGIC;
	config->header.length = sizeof(ring_config);
	config->header.version = COCOA_RING_CONFIG_VERSION;

	size_t sz = config->header.length - sizeof(config->checksum);
	uint32_t crc = crc32_wd(0, (unsigned char*)config, sz, CHUNKSZ_CRC32);

	// Set checksum
	config->checksum = crc;

	err = ubi_volume_write(COCOA_UBI_CONFIG_VOLUME_NAME, (char *)config, sizeof(ring_config));
	if (err) {
		printf("ring config write failure (%d)\n", err);
	}
	return err;
}
